package eu.europa.ec.taxud.cesop.validation;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Optional;
import java.util.Set;

import eu.europa.ec.taxud.cesop.domain.DocTypeEnum;
import eu.europa.ec.taxud.cesop.domain.MessageTypeIndicEnum;
import eu.europa.ec.taxud.cesop.domain.PaymentDataMsgPartContentType;
import eu.europa.ec.taxud.cesop.domain.ValidationError;
import eu.europa.ec.taxud.cesop.domain.XmlCountryTypeAndValue;
import eu.europa.ec.taxud.cesop.domain.XmlMessageSpec;
import eu.europa.ec.taxud.cesop.domain.XmlPaymentDataMsgPart;
import eu.europa.ec.taxud.cesop.domain.XmlPsp;
import eu.europa.ec.taxud.cesop.domain.XmlReportedPayee;
import eu.europa.ec.taxud.cesop.domain.XmlReportedTransaction;
import eu.europa.ec.taxud.cesop.domain.XmlTypeAndValue;
import eu.europa.ec.taxud.cesop.readers.IPspXmlReader;

import static eu.europa.ec.taxud.cesop.utils.ValidationErrorUtils.checkCmBr0010;
import static eu.europa.ec.taxud.cesop.utils.ValidationErrorUtils.checkCmBr0050;
import static eu.europa.ec.taxud.cesop.utils.ValidationErrorUtils.checkCmBr0060;
import static eu.europa.ec.taxud.cesop.utils.ValidationErrorUtils.checkCmBr0100;
import static eu.europa.ec.taxud.cesop.utils.ValidationErrorUtils.checkCmBr0110;
import static eu.europa.ec.taxud.cesop.utils.ValidationErrorUtils.checkCmBr0120;
import static eu.europa.ec.taxud.cesop.utils.ValidationErrorUtils.checkCmBr0130;
import static eu.europa.ec.taxud.cesop.utils.ValidationErrorUtils.checkCmBr0140;
import static eu.europa.ec.taxud.cesop.utils.ValidationErrorUtils.checkCmBr0140ForDates;
import static eu.europa.ec.taxud.cesop.utils.ValidationErrorUtils.checkCmBr0150;
import static eu.europa.ec.taxud.cesop.utils.ValidationErrorUtils.checkCmTr0070;
import static eu.europa.ec.taxud.cesop.utils.ValidationErrorUtils.checkMhBr0030;
import static eu.europa.ec.taxud.cesop.utils.ValidationErrorUtils.checkMhBr0070AndMhBr0080;
import static eu.europa.ec.taxud.cesop.utils.ValidationErrorUtils.checkMhBr0090;
import static eu.europa.ec.taxud.cesop.utils.ValidationErrorUtils.checkMhBr0110;
import static eu.europa.ec.taxud.cesop.utils.ValidationErrorUtils.checkRpBr0010;
import static eu.europa.ec.taxud.cesop.utils.ValidationErrorUtils.checkRpBr0020;
import static eu.europa.ec.taxud.cesop.utils.ValidationErrorUtils.checkRpBr0030;
import static eu.europa.ec.taxud.cesop.utils.ValidationErrorUtils.checkRpBr0040;
import static eu.europa.ec.taxud.cesop.utils.ValidationErrorUtils.checkRpBr0050;
import static eu.europa.ec.taxud.cesop.utils.ValidationErrorUtils.checkRpBr0060;
import static eu.europa.ec.taxud.cesop.utils.ValidationErrorUtils.checkRpBr0070;
import static eu.europa.ec.taxud.cesop.utils.ValidationErrorUtils.checkRpBr0080;
import static eu.europa.ec.taxud.cesop.utils.ValidationErrorUtils.checkRpBr0090;
import static eu.europa.ec.taxud.cesop.utils.ValidationErrorUtils.checkRpBr0100;
import static eu.europa.ec.taxud.cesop.utils.ValidationErrorUtils.checkRpBr0110;
import static eu.europa.ec.taxud.cesop.utils.ValidationErrorUtils.checkRtBr0010;
import static eu.europa.ec.taxud.cesop.utils.ValidationErrorUtils.checkRtBr0030;
import static eu.europa.ec.taxud.cesop.utils.ValidationErrorUtils.checkRtBr0040;
import static eu.europa.ec.taxud.cesop.utils.ValidationErrorUtils.checkRtBr0060;
import static eu.europa.ec.taxud.cesop.utils.ValidationErrorUtils.checkRtBr0080;
import static eu.europa.ec.taxud.cesop.utils.ValidationErrorUtils.checkRtBr0090;

/**
 * Stateful wrapper for PSP XML reader. Performs static validations during reading.
 */
public class PspValidatingReader {
    private final IPspXmlReader xmlReader;
    private final ValidationSettings settings;
    private final List<ValidationError> validationErrors;
    private final Set<String> transactionIdentifiersSet;
    private final Set<String> docRefSet;
    private final HashMap<Set<XmlTypeAndValue>, String> reportedPayeeSet;
    private final Set<String> corDocRefIds;
    private final boolean checkBRs;

    /**
     * Instantiates a new Psp validating reader.
     *
     * @param xmlReader the xml reader
     * @param settings  the settings
     * @param checkBRs  if true validate BR checks
     */
    public PspValidatingReader(final IPspXmlReader xmlReader, final ValidationSettings settings, boolean checkBRs, Set<String> corDocRefIds) {
        this.settings = settings;
        this.xmlReader = xmlReader;
        this.validationErrors = new CesopValidationErrorList(settings.getMaxErrors());
        this.transactionIdentifiersSet = new HashSet<>();
        this.docRefSet = new HashSet<>();
        this.reportedPayeeSet = new HashMap<>();
        this.corDocRefIds = corDocRefIds;
        this.checkBRs = checkBRs;
        if (checkBRs) {
            this.validateHeader();
        }
    }

    private void validateHeader() {
        final XmlMessageSpec xmlMessageSpec = this.xmlReader.getXmlMessageSpec();
        final String messageRefId = xmlMessageSpec.getMessageRefId();
        checkMhBr0030(this.validationErrors, messageRefId, xmlMessageSpec.getReportingPeriod(), this.settings.getBusinessMinPeriod());
        checkCmTr0070(this.validationErrors, messageRefId, this.xmlReader.getEstimatedContentSize(), this.settings.getMaxSize());
        checkMhBr0110(this.validationErrors, messageRefId, xmlMessageSpec.getCorrMessageRefId(), xmlMessageSpec.getMessageTypeIndic());
        checkMhBr0090(this.validationErrors, messageRefId, xmlMessageSpec.getMessageType(), this.xmlReader.getXmlReportingPsp() != null);
    }

    /**
     * Gets validation errors.
     *
     * @return the validation errors
     */
    public List<ValidationError> getValidationErrors() {
        return this.validationErrors;
    }

    /**
     * Validate parts iterator.
     * This iterator extracts and validates the list of {@link XmlPaymentDataMsgPart} from the XML content.
     *
     * @return the iterator
     */
    public Iterator<XmlPaymentDataMsgPart> validateParts() {
        return new Iterator<XmlPaymentDataMsgPart>() {

            int errorsBeforeCurrentPayee;
            boolean emptyTransactionsCurrentPayee = true;

            {
                if (checkBRs) {
                    MessageTypeIndicEnum messageTypeIndic = xmlReader.getXmlMessageSpec().getMessageTypeIndic();
                    String messageRefId = xmlReader.getXmlMessageSpec().getMessageRefId();
                    if (xmlReader.getXmlReportingPsp() != null) {
                        checkCmBr0100(validationErrors, messageRefId, xmlReader.getXmlReportingPsp().getPspId());
                        checkCmBr0130(validationErrors, messageRefId, xmlReader.getXmlReportingPsp().getPspId().getType(),
                                xmlReader.getXmlReportingPsp().getPspId().getOther(), xmlReader.getXmlMessageSpec().getXsdVersion());
                        if (hasNext()) {
                            checkRpBr0040(validationErrors, messageRefId, true, messageTypeIndic);
                        } else {
                            XmlMessageSpec messageSpec = xmlReader.getXmlMessageSpec();
                            checkCmBr0110(validationErrors, messageSpec.getMessageRefId(), messageSpec.getMessageTypeIndic());
                        }
                    }
                    XmlPsp sendingPsp = xmlReader.getXmlMessageSpec().getSendingPsp();
                    if (sendingPsp != null && sendingPsp.getPspId() != null) {
                        checkCmBr0130(validationErrors, messageRefId, sendingPsp.getPspId().getType(),
                                sendingPsp.getPspId().getOther(), xmlReader.getXmlMessageSpec().getXsdVersion());
                    }
                }
                errorsBeforeCurrentPayee = validationErrors.size();
            }

            @Override
            public boolean hasNext() {
                return xmlReader.hasNext();
            }

            @Override
            public XmlPaymentDataMsgPart next() {
                final XmlPaymentDataMsgPart next = xmlReader.next();
                if (next.getContentType() == PaymentDataMsgPartContentType.REPORTED_TRANSACTIONS) {
                    final int errCount = validationErrors.size();
                    emptyTransactionsCurrentPayee = next.getXmlReportedTransactions().isEmpty();
                    if (checkBRs) {
                        validateReportedTransactions(next);
                    }
                    if (validationErrors.size() > errCount) {
                        next.setError(true);
                    }
                } else {
                    if (checkBRs) {
                        validateReportedPayee(next, emptyTransactionsCurrentPayee);
                    }
                    if (validationErrors.size() > errorsBeforeCurrentPayee) {
                        next.setError(true);
                    }
                    final String docRefId = next.getXmlReportedPayee().getDocSpec().getDocRefId();
                    // We obtain docRefId only in the end of payee section, so we need to go back and set it for validation errors produced for this payee.
                    validationErrors.stream().skip(errorsBeforeCurrentPayee).forEach(err -> err.setDocRefId(docRefId));
                    errorsBeforeCurrentPayee = validationErrors.size();
                    emptyTransactionsCurrentPayee = true;
                }
                return next;
            }
        };
    }

    private void validateReportedPayee(final XmlPaymentDataMsgPart paymentDataMsgPart, final boolean emptyTransactions) {
        final String messageRefId = xmlReader.getXmlMessageSpec().getMessageRefId();
        final MessageTypeIndicEnum messageTypeIndic = xmlReader.getXmlMessageSpec().getMessageTypeIndic();

        final XmlReportedPayee xmlReportedPayee = paymentDataMsgPart.getXmlReportedPayee();
        final DocTypeEnum docType = DocTypeEnum.findByCode(xmlReportedPayee.getDocSpec().getDocTypeIndic());
        final String docRefId = xmlReportedPayee.getDocSpec().getDocRefId();
        final List<XmlCountryTypeAndValue> accounts = xmlReportedPayee.getAccountIdentifiers();
        final String xsdVersion = this.xmlReader.getXmlMessageSpec().getXsdVersion();

        accounts.forEach(account -> {
            checkRpBr0060(validationErrors, account, messageRefId);
            checkCmBr0140(validationErrors, messageRefId, account.getType(), account.getOther(), xsdVersion);
            checkRpBr0110(validationErrors, messageRefId, account.getType(), account.getOther(), xsdVersion);
        });

        checkRpBr0100(validationErrors, accounts, messageRefId, docRefId, xsdVersion);

        checkCmBr0150(validationErrors, messageRefId, reportedPayeeSet, xmlReportedPayee.getNames(), accounts, docRefId);
        checkRpBr0050(validationErrors, messageRefId, docType, emptyTransactions);
        checkCmBr0010(validationErrors, messageRefId, docRefSet, docRefId);
        checkMhBr0070AndMhBr0080(validationErrors, messageRefId, messageTypeIndic, docType, docRefId);
        final String corrDocRefId = xmlReportedPayee.getDocSpec().getCorrDocRefId();
        checkCmBr0050(validationErrors, messageRefId, docType, corrDocRefId, docRefId);
        checkCmBr0060(validationErrors, messageRefId, docType, corrDocRefId, docRefId);
        checkCmBr0120(validationErrors, messageRefId, docType, corrDocRefId, this.corDocRefIds, xsdVersion, docRefId);
        accounts.forEach(account -> {
            if (checkRpBr0020(validationErrors, messageRefId, account.getType(), account.getValue(), docRefId)) {
                checkRpBr0030(validationErrors, messageRefId, account.getType(), account.getValue(), docRefId);
            }
        });

        final XmlPsp representative = xmlReportedPayee.getRepresentative();
        if (representative != null) {
            checkRpBr0070(validationErrors, messageRefId, representative.getPspId(), docRefId);
            checkCmBr0140(validationErrors, messageRefId, representative.getPspId().getType(),representative.getPspId().getOther(), xsdVersion);
        }

        XmlCountryTypeAndValue accountIdentifier = accounts.stream().filter(v -> v.getValue() != null && !v.getValue().isEmpty())
                .findFirst()
                .orElse(null);
        checkRpBr0080(validationErrors, messageRefId, accountIdentifier, representative, docRefId);
        checkRpBr0090(validationErrors, messageRefId, docType, emptyTransactions, docRefId, xsdVersion);

        if (xmlReportedPayee.getTaxIdentifications() != null) {
            xmlReportedPayee.getTaxIdentifications().forEach(t -> checkCmBr0140(validationErrors, messageRefId, t.getType(), t.getOther(), xsdVersion));
        }
    }

    private void validateReportedTransactions(final XmlPaymentDataMsgPart paymentDataMsgPart) {
        final String messageRefId = this.xmlReader.getXmlMessageSpec().getMessageRefId();
        final Integer reportingPeriod = this.xmlReader.getXmlMessageSpec().getReportingPeriod();
        final String xsdVersion = xmlReader.getXmlMessageSpec().getXsdVersion();
        final List<XmlReportedTransaction> xmlReportedTransactions = paymentDataMsgPart.getXmlReportedTransactions();
        for (final XmlReportedTransaction xmlReportedTransaction : xmlReportedTransactions) {
            final String transactionIdentifier = xmlReportedTransaction.getTransactionIdentifier();
            checkRpBr0010(this.validationErrors, messageRefId, paymentDataMsgPart.getXmlReportedPayee().getCountry(), xmlReportedTransaction.getPayerMs().getType(), transactionIdentifier);
            checkRtBr0030(this.validationErrors, messageRefId, reportingPeriod, xmlReportedTransaction.getDates(), transactionIdentifier);
            checkRtBr0010(this.validationErrors, messageRefId, xmlReportedTransaction.isRefund(), xmlReportedTransaction.getAmount(), transactionIdentifier);
            checkRtBr0040(this.validationErrors, messageRefId, this.transactionIdentifiersSet, transactionIdentifier);
            checkRtBr0060(this.validationErrors, messageRefId, xmlReportedTransaction.getAmount().getValue(), transactionIdentifier);
            checkRtBr0080(this.validationErrors, messageRefId, xmlReportedTransaction.isInErrorRtBr0080(), transactionIdentifier);
            checkCmBr0140(this.validationErrors, messageRefId, xmlReportedTransaction.getPaymentMethodType(),
                    Optional.ofNullable(xmlReportedTransaction.getPaymentMethodOtherId()).map(Object::toString).orElse(null), xsdVersion);
            checkCmBr0130(this.validationErrors, messageRefId, xmlReportedTransaction.getPspRoleType(),
                    Optional.ofNullable(xmlReportedTransaction.getPspRoleOtherId()).map(Object::toString).orElse(null), xsdVersion);
            checkCmBr0140ForDates(validationErrors, messageRefId, xmlReportedTransaction.getDates(), xsdVersion);
            checkRtBr0090(validationErrors, messageRefId, xmlReportedTransaction.isRefund(), transactionIdentifier, xmlReportedTransaction.getCorrTransactionIdentifier());
        }
    }

    public int getNumberOfTransaction() {
        return this.transactionIdentifiersSet.size();
    }

    public int getNumberOfPayees() {
        return this.docRefSet.size();
    }

    public XmlMessageSpec getXmlMessageSpec() {
        return this.xmlReader.getXmlMessageSpec();
    }

}
